// public/_worker.js (Packet-Ready Template - V11 Digital Download Integrated)

// --- Configuration & Constants ---
const DEFAULT_PEPEBLOCKS_API_BASE = "https://pepeblocks.com/api/v2";
const DEFAULT_SESSION_LOCK_DURATION_SECONDS = 30 * 60; // 30 minutes
const DEFAULT_MIN_CONFIRMATIONS = 6;
const DEFAULT_DOWNLOAD_TOKEN_TTL_SECONDS = 24 * 60 * 60; // 24 hours for download link
const SITE_LOCK_KEY = "CURRENT_SITE_LOCK";
const DEFAULT_DIGITAL_PRODUCT_FILENAME = "thank_you.zip"; // User can change this via ENV or keep it as it is

// --- Helper: CORS Headers ---
function addCorsHeaders(response) { 
    response.headers.set('Access-Control-Allow-Origin', '*');
    response.headers.set('Access-Control-Allow-Methods', 'GET, POST, OPTIONS');
    response.headers.set('Access-Control-Allow-Headers', 'Content-Type, Authorization');
    return response;
}
// --- Helper: Get Environment Variable with Default ---
function getEnv(env, key, defaultValue) { 
    const value = env[key];
    if (value === undefined || value === null || value === "") {
        return defaultValue;
    }
    return value;
}
// --- Helper: Generate UUID ---
function generateUUID() { 
    return crypto.randomUUID();
}
// --- Helper: JSON Response ---
function jsonResponse(data, status = 200) { 
    return new Response(JSON.stringify(data), {
        status: status,
        headers: { 'Content-Type': 'application/json' }
    });
}

// --- Main Fetch Handler ---
export default {
    async fetch(request, env, ctx) {
        const url = new URL(request.url);
        const method = request.method;
        let pathname = url.pathname; // Use let for modification
        const logPrefix = "[PEP_STORE_TEMPLATE_V11]";
        const workerUserAgent = getEnv(env, 'WORKER_USER_AGENT', 'PepStoreTemplateWorker/1.1');

        // Essential KVs now include DOWNLOAD_ACCESS_TOKENS
        const essentialKVs = ['SITE_WIDE_LOCK', 'USER_ORDER_ATTEMPTS', 'USER_PROFILES', 'PAID_ORDERS_ARCHIVE', 'PROCESSED_TXS_GLOBAL', 'DOWNLOAD_ACCESS_TOKENS'];
        // Check KVs for API routes that require them
        if (["/initiate-payment-session", "/verify-payment", "/access-paid-order", "/get-user-status"].includes(pathname) || pathname.startsWith("/download-digital/")) {
            for (const rk of essentialKVs) {
                if (!env[rk] && !(rk === 'DOWNLOAD_ACCESS_TOKENS' && !pathname.startsWith("/download-digital/") && pathname !== "/verify-payment" && pathname !== "/access-paid-order")) { // DOWNLOAD_ACCESS_TOKENS only strictly needed by some
                    console.error(`${logPrefix} CRITICAL_ERROR: KV binding ${rk} missing for path ${pathname}.`);
                    return addCorsHeaders(jsonResponse({ error: `Server misconfiguration (KV ${rk} not bound).` }, 503));
                }
            }
        }
        
        if (method === "OPTIONS") {
            if (["/initiate-payment-session", "/verify-payment", "/access-paid-order", "/get-user-status"].includes(pathname) || pathname.startsWith("/download-digital/")) {
                return addCorsHeaders(new Response(null, { status: 204 }));
            }
        }

        try {
            if (pathname === "/initiate-payment-session" && method === "POST") {
                console.log(`${logPrefix} Routing to /initiate-payment-session`);
                return addCorsHeaders(await handleInitiatePaymentSession(request, env, ctx, logPrefix));
            }
            if (pathname === "/verify-payment" && method === "POST") {
                console.log(`${logPrefix} Routing to /verify-payment`);
                return addCorsHeaders(await handleVerifyPayment(request, env, ctx, logPrefix, workerUserAgent));
            }
            if (pathname === "/access-paid-order" && method === "POST") {
                console.log(`${logPrefix} Routing to /access-paid-order`);
                return addCorsHeaders(await handleAccessPaidOrder(request, env, ctx, logPrefix, workerUserAgent));
            }
            if (pathname === "/get-user-status" && method === "GET") {
                console.log(`${logPrefix} Routing to /get-user-status`);
                return addCorsHeaders(await handleGetUserStatus(request, env, ctx, logPrefix));
            }
            // Digital Download Endpoint
            if (pathname.startsWith("/download-digital/") && method === "GET") {
                console.log(`${logPrefix} Routing to /download-digital`);
                const token = pathname.substring("/download-digital/".length);
                // No CORS headers needed for direct file download usually, but errors might benefit
                return handleDigitalDownload(request, env, ctx, logPrefix, token); 
            }
        } catch (e) { /* ... same error handling ... */ 
            console.error(`${logPrefix} UNHANDLED_TOP_LEVEL_API_ERROR on ${pathname}: Name: ${e.name}, Msg: ${e.message}`);
            return addCorsHeaders(jsonResponse({ error: "An unexpected server error occurred." }, 500));
        }

        if (env.ASSETS && typeof env.ASSETS.fetch === 'function') { /* ... same ... */ 
             try {
                let assetRequestPath = pathname;
                if (pathname === '/' || pathname.endsWith('/')) {
                    assetRequestPath = (pathname.endsWith('/') ? pathname : pathname + '/') + 'index.html';
                }
                const assetRequestUrl = new URL(assetRequestPath, request.url).toString();
                const assetRequest = new Request(assetRequestUrl, { method: "GET", headers: request.headers });
                return env.ASSETS.fetch(assetRequest);
            } catch (e) { console.error(`${logPrefix} Error during ASSETS.fetch for ${request.url}: ${e.message}`); }
        }
        return new Response("Resource not found.", { status: 404 });
    }
};

// handleInitiatePaymentSession remains the same
async function handleInitiatePaymentSession(request, env, ctx, parentLogPrefix) { /* ... same as V10 ... */ 
    const logPrefix = `${parentLogPrefix}[INIT_SESSION]`;
    const sessionLockDurationSeconds = parseInt(getEnv(env, 'SESSION_LOCK_DURATION_SECONDS', DEFAULT_SESSION_LOCK_DURATION_SECONDS.toString()));
    try {
        const { email, order_details } = await request.json();
        if (!email || !email.includes('@') || !order_details || !order_details.items || typeof order_details.total_pep_amount !== 'number') { // Removed items.length check as digital order might be 0 physical items from user view before adding
            return jsonResponse({ error: "Valid email and complete order details are required." }, 400);
        }
        const normalizedEmail = email.toLowerCase().trim();
        // Ensure order_details contains hasDigital/hasPhysical if frontend sends it, or derive it
        const hasPhysical = order_details.items.some(item => item.type === 'physical');
        const hasDigital = order_details.items.some(item => item.type === 'digital');
        const finalOrderDetails = {...order_details, hasPhysical, hasDigital};


        console.log(`${logPrefix} INPUT: Email='${normalizedEmail}', OrderTotal='${finalOrderDetails.total_pep_amount}', HasPhysical=${hasPhysical}, HasDigital=${hasDigital}`);

        const currentLockString = await env.SITE_WIDE_LOCK.get(SITE_LOCK_KEY);
        let currentLock = currentLockString ? JSON.parse(currentLockString) : null;
        const now = Date.now();

        if (currentLock && currentLock.lock_expiry_ts > now && currentLock.email !== normalizedEmail) {
            console.warn(`${logPrefix} SITE_LOCKED by other user: ${currentLock.email}. Expires: ${new Date(currentLock.lock_expiry_ts).toISOString()}`);
            const timeLeft = Math.max(1, Math.ceil((currentLock.lock_expiry_ts - now) / 60000));
            return jsonResponse({ error: `System busy. Try again in ~${timeLeft} min.` }, 429);
        }
        console.log(`${logPrefix} Site lock check passed.`);

        const order_attempt_id = generateUUID();
        const session_start_ts = now;
        const session_end_ts = session_start_ts + (sessionLockDurationSeconds * 1000);
        const attemptData = { email: normalizedEmail, order_attempt_id, session_start_ts, session_end_ts, order_details: finalOrderDetails, status: "awaiting_redemption" };
        
        ctx.waitUntil(env.USER_ORDER_ATTEMPTS.put(order_attempt_id, JSON.stringify(attemptData)));
        console.log(`${logPrefix} USER_ORDER_ATTEMPTS: PUT key='${order_attempt_id}'`);

        let userProfile = await env.USER_PROFILES.get(normalizedEmail).then(res => res ? JSON.parse(res) : null) || { pending_order_attempt_ids: [], paid_order_tx_ids: [] };
        if (!userProfile.pending_order_attempt_ids.includes(order_attempt_id)) userProfile.pending_order_attempt_ids.push(order_attempt_id);
        ctx.waitUntil(env.USER_PROFILES.put(normalizedEmail, JSON.stringify(userProfile)));
        console.log(`${logPrefix} USER_PROFILES: Updated for '${normalizedEmail}'.`);

        const newLock = { email: normalizedEmail, order_attempt_id, lock_expiry_ts: session_end_ts };
        ctx.waitUntil(env.SITE_WIDE_LOCK.put(SITE_LOCK_KEY, JSON.stringify(newLock)));
        console.log(`${logPrefix} SITE_WIDE_LOCK: SET for '${normalizedEmail}', attempt '${order_attempt_id}', expires '${new Date(session_end_ts).toISOString()}'`);

        return jsonResponse({
            status: "session_initiated",
            message: `Session locked. Pay ${finalOrderDetails.total_pep_amount} $PEP within ${sessionLockDurationSeconds / 60} minutes.`,
            order_attempt_id: order_attempt_id
        });
    } catch (e) { 
        console.error(`${logPrefix} UNCAUGHT_ERROR: Name: ${e.name}, Msg: ${e.message}`);
        return jsonResponse({ error: "Error initiating session." }, 500);
    }
}


// MODIFIED handleVerifyPayment to handle digital download token generation
async function handleVerifyPayment(request, env, ctx, parentLogPrefix, workerUserAgent) {
    const logPrefix = `${parentLogPrefix}[VERIFY_PAY]`;
    // ... (initial env var checks and request parsing - same as V10) ...
    const receivingAddress = getEnv(env, 'RECEIVING_ADDRESS', null);
    if (!receivingAddress) return jsonResponse({ error: "Server Misconfig: RECEIVING_ADDRESS" }, 503);
    const minConfirmations = parseInt(getEnv(env, 'MIN_CONFIRMATIONS', DEFAULT_MIN_CONFIRMATIONS.toString()));
    const pepeblocksApiBase = getEnv(env, 'PEPEBLOCKS_API_BASE', DEFAULT_PEPEBLOCKS_API_BASE);
    const downloadTokenTTLSeconds = parseInt(getEnv(env, 'DOWNLOAD_TOKEN_TTL_SECONDS', DEFAULT_DOWNLOAD_TOKEN_TTL_SECONDS.toString()));
    const digitalProductFilename = getEnv(env, 'DIGITAL_PRODUCT_FILENAME', DEFAULT_DIGITAL_PRODUCT_FILENAME);


    try {
        const { email, tx_hash, order_attempt_id } = await request.json();
        if (!email || !tx_hash || !order_attempt_id) return jsonResponse({ error: "Email, TX Hash, and Order Attempt ID required." }, 400);
        const normalizedEmail = email.toLowerCase().trim();
        console.log(`${logPrefix} INPUT: Email='${normalizedEmail}', TX='${tx_hash}', AttemptID='${order_attempt_id}'`);

        if (await env.PROCESSED_TXS_GLOBAL.get(tx_hash)) return jsonResponse({ error: "TXID already redeemed." }, 409);

        const attemptString = await env.USER_ORDER_ATTEMPTS.get(order_attempt_id);
        if (!attemptString) return jsonResponse({ error: "Order attempt not found/expired." }, 404);
        const attemptData = JSON.parse(attemptString);
        if (attemptData.email !== normalizedEmail) return jsonResponse({ error: "Email mismatch for order attempt." }, 403);
        if (attemptData.status === "redeemed") return jsonResponse({ error: "Order attempt already redeemed." }, 409);

        const { session_start_ts, session_end_ts, order_details } = attemptData; // order_details here includes hasDigital/hasPhysical
        const expectedTotalRibbits = order_details.total_pep_amount * 100000000;
        console.log(`${logPrefix} Attempt '${order_attempt_id}': Expected ${order_details.total_pep_amount} PEP. Window: ${new Date(session_start_ts).toISOString()} to ${new Date(session_end_ts).toISOString()}`);

        const txDetailResponse = await fetch(`${pepeblocksApiBase}/tx/${tx_hash}`, { headers: { 'User-Agent': workerUserAgent } });
        if (!txDetailResponse.ok) return jsonResponse({ error: `TX API Error (${txDetailResponse.status}).` }, 502);
        const txData = await txDetailResponse.json();
        console.log(`${logPrefix} TX API Data for ${tx_hash}: Confirmations=${txData.confirmations}, Time=${txData.blockTime || txData.time}`);
        
        const confirmations = txData.confirmations || 0;
        if (confirmations < minConfirmations) return jsonResponse({ error: `Need ${minConfirmations} confirmations (have ${confirmations}).` }, 400);
        
        const blockchainTxTs_API = txData.blockTime || txData.time;
        if (!blockchainTxTs_API || isNaN(blockchainTxTs_API)) return jsonResponse({ error: "TX missing valid timestamp from API." }, 422);
        const blockchainTxTs_ms = blockchainTxTs_API * 1000;
        if (!(blockchainTxTs_ms >= session_start_ts && blockchainTxTs_ms <= session_end_ts)) return jsonResponse({ error: "TX timestamp outside payment window." }, 400);
        
        let paidToUsAmountRibbits = 0; let paymentToUsFound = false;
        (txData.vout || []).forEach(vout => { if (vout.addresses && vout.addresses.includes(receivingAddress)) { paymentToUsFound = true; paidToUsAmountRibbits += parseInt(vout.value || '0'); }});
        if (!paymentToUsFound) return jsonResponse({ error: "Payment not to correct address." }, 400);
        if (paidToUsAmountRibbits < expectedTotalRibbits) return jsonResponse({ error: `Paid amount insufficient.` }, 400);
        
        console.log(`${logPrefix} ALL CHECKS PASSED for TX '${tx_hash}', Email '${normalizedEmail}', Attempt '${order_attempt_id}'.`);
        const redeemed_at_ts = Date.now();
        const processedTxData = { email: normalizedEmail, redeemed_at_ts, associated_order_attempt_id: order_attempt_id, order_total_pep: order_details.total_pep_amount };
        // The order_details in attemptData already contains item types
        const paidOrderArchiveData = { email: normalizedEmail, original_order_attempt_id: order_attempt_id, redeemed_order_details: order_details, blockchain_tx_ts: blockchainTxTs_ms, redeemed_at_ts, paid_amount_ribbits: paidToUsAmountRibbits, session_start_ts_of_order: session_start_ts };
        
        const updatePromises = [];
        updatePromises.push(env.PROCESSED_TXS_GLOBAL.put(tx_hash, JSON.stringify(processedTxData)));
        updatePromises.push(env.PAID_ORDERS_ARCHIVE.put(tx_hash, JSON.stringify(paidOrderArchiveData)));
        attemptData.status = "redeemed"; attemptData.redeemed_tx_hash = tx_hash;
        updatePromises.push(env.USER_ORDER_ATTEMPTS.put(order_attempt_id, JSON.stringify(attemptData)));
        const updateUserProfile = async () => { /* ... same as V10 ... */ 
            let userProfile = await env.USER_PROFILES.get(normalizedEmail).then(res => res ? JSON.parse(res) : null) || { pending_order_attempt_ids: [], paid_order_tx_ids: [] };
            userProfile.pending_order_attempt_ids = userProfile.pending_order_attempt_ids.filter(id => id !== order_attempt_id);
            if (!userProfile.paid_order_tx_ids.includes(tx_hash)) userProfile.paid_order_tx_ids.push(tx_hash);
            await env.USER_PROFILES.put(normalizedEmail, JSON.stringify(userProfile));
        };
        updatePromises.push(updateUserProfile());
        const clearSiteLock = async () => { /* ... same as V10 ... */
            const currentLockString = await env.SITE_WIDE_LOCK.get(SITE_LOCK_KEY);
            if (currentLockString) { const cl = JSON.parse(currentLockString); if (cl.order_attempt_id === order_attempt_id) await env.SITE_WIDE_LOCK.delete(SITE_LOCK_KEY); }
        };
        updatePromises.push(clearSiteLock());

        let download_token = null;
        if (order_details.hasDigital) { // Check if the original order had digital items
            download_token = generateUUID();
            const downloadTokenData = { 
                tx_id: tx_hash, 
                email: normalizedEmail, 
                filename_to_serve: digitalProductFilename // Use configured filename
            };
            updatePromises.push(env.DOWNLOAD_ACCESS_TOKENS.put(download_token, JSON.stringify(downloadTokenData), { expirationTtl: downloadTokenTTLSeconds }));
            console.log(`${logPrefix} Generated download_token '${download_token}' for digital item in TX '${tx_hash}'.`);
        }
        
        ctx.waitUntil(Promise.all(updatePromises));
        console.log(`${logPrefix} KV Updates Scheduled for TX '${tx_hash}'.`);

        return jsonResponse({
            status: "payment_verified_and_linked",
            message: "Payment successfully verified! Order confirmed.",
            order_details_confirmed: { items: order_details.items, shipping_fee: order_details.shipping_fee, total_pep_amount: order_details.total_pep_amount, hasDigital: order_details.hasDigital, hasPhysical: order_details.hasPhysical }, // Pass these flags
            session_details: { session_start_ts: session_start_ts },
            tx_id: tx_hash,
            download_token: download_token // Will be null if no digital items
        });
    } catch (e) { /* ... same error handling ... */ 
        console.error(`${logPrefix} UNCAUGHT_ERROR: Name: ${e.name}, Msg: ${e.message}`);
        return jsonResponse({ error: "Error during payment verification." }, 500);
    }
}

// MODIFIED handleAccessPaidOrder to handle digital download token generation for re-access
async function handleAccessPaidOrder(request, env, ctx, parentLogPrefix, workerUserAgent) {
    const logPrefix = `${parentLogPrefix}[ACCESS_OR_REDEEM_V11]`; // Updated prefix
    // ... (initial env var checks - same as V10, ensure DOWNLOAD_ACCESS_TOKENS and relevant env vars are checked if used) ...
    const receivingAddress = getEnv(env, 'RECEIVING_ADDRESS', null); // Needed if falling back to redeem
    if (!receivingAddress && ! (await env.PAID_ORDERS_ARCHIVE.get(JSON.parse(await request.clone().text()).tx_id)) ) { // Only strictly need if we might redeem
         // A bit complex to check if it will be needed; assume it might be for now or simplify error handling
    }
    const minConfirmations = parseInt(getEnv(env, 'MIN_CONFIRMATIONS', DEFAULT_MIN_CONFIRMATIONS.toString()));
    const pepeblocksApiBase = getEnv(env, 'PEPEBLOCKS_API_BASE', DEFAULT_PEPEBLOCKS_API_BASE);
    const downloadTokenTTLSeconds = parseInt(getEnv(env, 'DOWNLOAD_TOKEN_TTL_SECONDS', DEFAULT_DOWNLOAD_TOKEN_TTL_SECONDS.toString()));
    const digitalProductFilename = getEnv(env, 'DIGITAL_PRODUCT_FILENAME', DEFAULT_DIGITAL_PRODUCT_FILENAME);


    try {
        const { email, tx_id } = await request.json();
        if (!email || !tx_id) return jsonResponse({ error: "Email and Transaction ID required." }, 400);
        const normalizedEmail = email.toLowerCase().trim();
        console.log(`${logPrefix} Attempt for Email: '${normalizedEmail}', TX_ID: '${tx_id}'`);

        const paidOrderString = await env.PAID_ORDERS_ARCHIVE.get(tx_id);
        if (paidOrderString) {
            const paidOrderData = JSON.parse(paidOrderString);
            if (paidOrderData.email === normalizedEmail) {
                console.log(`${logPrefix} DIRECT_MATCH in PAID_ORDERS_ARCHIVE. Granting access.`);
                let download_token = null;
                // Check if the redeemed order had digital items and generate a new download token
                if (paidOrderData.redeemed_order_details && paidOrderData.redeemed_order_details.hasDigital) {
                    download_token = generateUUID();
                    const downloadTokenData = { tx_id: tx_id, email: normalizedEmail, filename_to_serve: digitalProductFilename };
                    ctx.waitUntil(env.DOWNLOAD_ACCESS_TOKENS.put(download_token, JSON.stringify(downloadTokenData), { expirationTtl: downloadTokenTTLSeconds }));
                    console.log(`${logPrefix} Generated new download_token '${download_token}' for re-access of digital item in TX '${tx_id}'.`);
                }
                return jsonResponse({
                    status: "linked_order_access_granted", message: "Accessing previously confirmed order.",
                    order_details_confirmed: paidOrderData.redeemed_order_details, // This already includes hasDigital/hasPhysical
                    session_details: { session_start_ts: paidOrderData.session_start_ts_of_order },
                    tx_id: tx_id,
                    download_token: download_token
                });
            } else { return jsonResponse({ error: "TXID associated with a different account." }, 403); }
        }
        console.log(`${logPrefix} TX_ID '${tx_id}' not in PAID_ORDERS_ARCHIVE. Checking PROCESSED_TXS_GLOBAL.`);
        if (await env.PROCESSED_TXS_GLOBAL.get(tx_id)) return jsonResponse({ error: "TXID processed but not linked to your query." }, 409);
        
        console.log(`${logPrefix} TX_ID '${tx_id}' not paid/globally processed. Attempting to redeem against pending for '${normalizedEmail}'.`);
        const userProfileString = await env.USER_PROFILES.get(normalizedEmail);
        if (!userProfileString) return jsonResponse({ error: "No order history for email." }, 404);
        const userProfile = JSON.parse(userProfileString);
        if (!userProfile.pending_order_attempt_ids || userProfile.pending_order_attempt_ids.length === 0) return jsonResponse({ error: "No pending orders to match TXID." }, 404);

        // --- Fallback to redeem logic (simplified: find first match) ---
        const txDetailResponse = await fetch(`${pepeblocksApiBase}/tx/${tx_id}`, { headers: { 'User-Agent': workerUserAgent } });
        if (!txDetailResponse.ok) return jsonResponse({ error: `Could not get TX details (API Status: ${txDetailResponse.status}).` }, 502);
        const txDataFromBlockchain = await txDetailResponse.json();
        const confirmations = txDataFromBlockchain.confirmations || 0;
        if (confirmations < minConfirmations) return jsonResponse({ error: `TX needs ${minConfirmations} confirmations (has ${confirmations}).` }, 400);
        const blockchainTxTs_API = txDataFromBlockchain.blockTime || txDataFromBlockchain.time;
        if (!blockchainTxTs_API || isNaN(blockchainTxTs_API)) return jsonResponse({ error: "TX missing valid timestamp from API." }, 422);
        const blockchainTxTs_ms = blockchainTxTs_API * 1000;

        for (const attempt_id of userProfile.pending_order_attempt_ids) {
            const attemptString = await env.USER_ORDER_ATTEMPTS.get(attempt_id);
            if (!attemptString) continue;
            const currentAttemptData = JSON.parse(attemptString);
            if (currentAttemptData.status === "redeemed" || currentAttemptData.email !== normalizedEmail) continue;

            const { session_start_ts, session_end_ts, order_details } = currentAttemptData;
            const expectedTotalRibbits = order_details.total_pep_amount * 100000000;

            if (!(blockchainTxTs_ms >= session_start_ts && blockchainTxTs_ms <= session_end_ts)) continue;
            let paidToUsAmountRibbits = 0; let paymentToUsFound = false;
            (txDataFromBlockchain.vout || []).forEach(vout => { if (vout.addresses && vout.addresses.includes(receivingAddress)) { paymentToUsFound = true; paidToUsAmountRibbits += parseInt(vout.value || '0'); }});
            if (!paymentToUsFound || paidToUsAmountRibbits < expectedTotalRibbits) continue;

            // Match found! Delegate to handleVerifyPayment logic
            console.log(`${logPrefix} MATCHED TX '${tx_id}' with pending attempt '${attempt_id}'. Delegating to verify...`);
            const mockRequest = { json: async () => ({ email: normalizedEmail, tx_hash: tx_id, order_attempt_id: attempt_id }), headers: request.headers, method: "POST", url: request.url };
            return handleVerifyPayment(mockRequest, env, ctx, logPrefix, workerUserAgent); // This will generate download_token if needed
        }
        
        console.log(`${logPrefix} No pending order attempt for '${normalizedEmail}' matched criteria for TX_ID '${tx_id}'.`);
        return jsonResponse({ error: "TXID does not match any pending orders (timing, amount, etc.) or lacks confirmations." }, 404);

    } catch (e) { /* ... same error handling ... */ 
        console.error(`${logPrefix} UNCAUGHT_ERROR: Name: ${e.name}, Msg: ${e.message}`);
        return jsonResponse({ error: "Error processing access/redeem request." }, 500);
    }
}

// NEW Endpoint: /download-digital/[token]
async function handleDigitalDownload(request, env, ctx, parentLogPrefix, token) {
    const logPrefix = `${parentLogPrefix}[DOWNLOAD]`;
    console.log(`${logPrefix} Attempting download with token: ${token}`);

    if (!token) {
        return new Response("Download token missing.", { status: 400 });
    }

    const tokenDataString = await env.DOWNLOAD_ACCESS_TOKENS.get(token);
    if (!tokenDataString) {
        console.log(`${logPrefix} Token ${token} not found or expired.`);
        return new Response("Invalid or expired download link. Links are one-time use.", { status: 404 });
    }
    
    // Delete token immediately to make it one-time use
    ctx.waitUntil(env.DOWNLOAD_ACCESS_TOKENS.delete(token));
    console.log(`${logPrefix} Token ${token} redeemed and deleted.`);

    const tokenData = JSON.parse(tokenDataString);
    const filenameToServe = tokenData.filename_to_serve || getEnv(env, 'DIGITAL_PRODUCT_FILENAME', DEFAULT_DIGITAL_PRODUCT_FILENAME);

    if (!filenameToServe) {
        console.error(`${logPrefix} No filename configured to serve for token ${token}.`);
        return new Response("Digital product file not configured on server.", { status: 500 });
    }

    try {
        // Serve the file from Pages assets
        const assetPath = `/${filenameToServe.startsWith('/') ? filenameToServe.substring(1) : filenameToServe}`;
        console.log(`${logPrefix} Attempting to serve asset: ${assetPath} for token ${token}`);
        
        const asset = await env.ASSETS.fetch(new URL(assetPath, request.url).toString());

        if (asset.ok) {
            const responseHeaders = new Headers(asset.headers);
            responseHeaders.set('Content-Disposition', `attachment; filename="${filenameToServe}"`);
            // If you know the content type and want to ensure it's correct:
            // responseHeaders.set('Content-Type', 'application/zip'); // Or appropriate MIME type
            
            console.log(`${logPrefix} Serving file ${filenameToServe} for token ${token}.`);
            return new Response(asset.body, {
                headers: responseHeaders,
                status: 200
            });
        } else {
            console.error(`${logPrefix} Asset ${assetPath} not found or fetch failed. Status: ${asset.status}`);
            return new Response(`Digital product '${filenameToServe}' not found on server. Status: ${asset.status}`, { status: asset.status });
        }
    } catch (e) {
        console.error(`${logPrefix} Error serving digital asset ${filenameToServe}:`, e.message, e.stack);
        return new Response("Error retrieving digital product.", { status: 500 });
    }
}


// handleGetUserStatus remains the same as V10
async function handleGetUserStatus(request, env, ctx, parentLogPrefix) { /* ... same as V10 ... */ 
    const logPrefix = `${parentLogPrefix}[USER_STATUS]`;
    const url = new URL(request.url);
    const email = url.searchParams.get("email");

    if (!email || !email.includes('@')) return jsonResponse({ error: "Valid email query param required." }, 400);
    const normalizedEmail = email.toLowerCase().trim();

    try {
        const userProfileString = await env.USER_PROFILES.get(normalizedEmail);
        if (!userProfileString) return jsonResponse({ message: "No history for this email." });
        const userProfile = JSON.parse(userProfileString);
        console.log(`${logPrefix} Profile for ${normalizedEmail}: Pending=${userProfile.pending_order_attempt_ids.length}, Paid=${userProfile.paid_order_tx_ids.length}`);

        let pending_attempts_summary = [];
        if (userProfile.pending_order_attempt_ids && userProfile.pending_order_attempt_ids.length > 0) {
            for (const attempt_id of userProfile.pending_order_attempt_ids) {
                const attemptString = await env.USER_ORDER_ATTEMPTS.get(attempt_id);
                if (attemptString) {
                    const attemptData = JSON.parse(attemptString);
                    if (attemptData.status === "awaiting_redemption") { 
                         pending_attempts_summary.push({
                            order_attempt_id: attempt_id,
                            initiated_on: new Date(attemptData.session_start_ts).toLocaleDateString(),
                            total_amount: attemptData.order_details.total_pep_amount,
                            item_hint: attemptData.order_details.items[0] ? `${attemptData.order_details.items[0].quantity} x ${attemptData.order_details.items[0].name.substring(0,20)}...` : 'Order',
                            payment_window_ended: new Date(attemptData.session_end_ts).toLocaleString(),
                            status: attemptData.status
                        });
                    }
                } else { console.warn(`${logPrefix} Orphaned pending_id ${attempt_id} in profile ${normalizedEmail}`); }
            }
        }

        let paid_orders_summary = [];
        if (userProfile.paid_order_tx_ids && userProfile.paid_order_tx_ids.length > 0) {
            for (const tx_id of userProfile.paid_order_tx_ids) {
                const paidOrderString = await env.PAID_ORDERS_ARCHIVE.get(tx_id);
                if (paidOrderString) {
                    const paidOrderData = JSON.parse(paidOrderString);
                    paid_orders_summary.push({
                        tx_id_full: tx_id,
                        paid_on: new Date(paidOrderData.blockchain_tx_ts).toLocaleDateString(),
                        total_amount: paidOrderData.redeemed_order_details.total_pep_amount,
                        item_hint: paidOrderData.redeemed_order_details.items[0] ? `${paidOrderData.redeemed_order_details.items[0].quantity} x ${paidOrderData.redeemed_order_details.items[0].name.substring(0,20)}...` : 'Order'
                    });
                } else { console.warn(`${logPrefix} Orphaned paid_id ${tx_id} in profile ${normalizedEmail}`); }
            }
        }
        
        return jsonResponse({
            email: normalizedEmail,
            pending_attempts: pending_attempts_summary,
            paid_orders: paid_orders_summary
        });
    } catch (e) { 
        console.error(`${logPrefix} UNCAUGHT_ERROR: Name: ${e.name}, Msg: ${e.message}`);
        return jsonResponse({ error: "Error fetching user status." }, 500);
    }
}